// Define all skills (should match curriculum.html)
const ALL_SKILLS = [
  // A0 Level
  'a0-present-tense', 'a0-essentials', 'a0-greetings',
  'a0-numbers', 'a0-articles', 'a0-questions',
  // A1 Level
  'a1-preterite', 'a1-adjectives', 'a1-daily-routine', 'a1-food',
  // A2 Level
  'a2-imperfect', 'a2-future', 'a2-por-para', 'a2-commands',
  // B1 Level
  'b1-subjunctive-present', 'b1-perfect-tenses', 'b1-conditional', 'b1-pronouns',
  // B2 Level
  'b2-subjunctive-past', 'b2-passive-voice', 'b2-idioms', 'b2-business',
  // C1 Level
  'c1-advanced-grammar', 'c1-literature', 'c1-regional', 'c1-debates',
  // C2 Level
  'c2-nuance', 'c2-academic', 'c2-specialized', 'c2-native-level'
];

class PageTranslator {
  constructor() {
    this.elements = [];
    this.applied = new Set();
    this.translations = new Map();
    this.fullContext = '';
    this.jobStarted = false;
    this.jobComplete = false;
    this.liveMode = false;
    this.isApplying = false;
    this.isTranslatingNew = false;
    this.observer = null;
    this.abortController = null;
    this.currentSession = null;
    this.seenTexts = new Set();
    this.basicSpanishPrompt = null;
  }

  async loadBasicSpanishPrompt() {
    if (this.basicSpanishPrompt) return this.basicSpanishPrompt;

    try {
      const url = chrome.runtime.getURL('basic_spanish_prompt.txt');
      const response = await fetch(url);
      this.basicSpanishPrompt = await response.text();
      return this.basicSpanishPrompt;
    } catch (error) {
      console.error('[Immerse] Failed to load basic Spanish prompt:', error);
      return null;
    }
  }

  async checkUserSkillsCompleted() {
    try {
      console.log('[Immerse] checkUserSkillsCompleted: Starting skill check...');

      // Get user data from storage
      let userData;
      try {
        userData = await chrome.storage.local.get('immerse_user');
      } catch (storageError) {
        console.error('[Immerse] checkUserSkillsCompleted: Chrome storage access failed:', storageError);
        return false;
      }

      const user = userData.immerse_user;

      if (!user || !user.id) {
        console.log('[Immerse] checkUserSkillsCompleted: No user data found in storage');
        return false;
      }

      // Dynamically get all A0 and A1 skills from the ALL_SKILLS list
      const requiredLevels = ['a0', 'a1'];
      const requiredSkills = ALL_SKILLS.filter(skill => {
        const skillLevel = skill.split('-')[0].toLowerCase();
        return requiredLevels.includes(skillLevel);
      });

      const completedSkills = user.completed_skills || [];

      // Check if all required skills are completed
      const allCompleted = requiredSkills.every(skill => completedSkills.includes(skill));

      console.log(`[Immerse] checkUserSkillsCompleted: User has completed ${completedSkills.length} skills. Required A0/A1: ${requiredSkills.length}. All basics completed: ${allCompleted}`);

      return allCompleted;
    } catch (error) {
      console.error('[Immerse] checkUserSkillsCompleted: Unexpected error:', error);
      return false;
    }
  }

  async buildPrompt(targetLang, targetLangCode, userLevel, elementsToTranslate) {
    console.log(`[Immerse] buildPrompt: Called with targetLang=${targetLang}, userLevel=${userLevel}, elements=${elementsToTranslate.length}`);

    try {
      // Check if user has completed all A0 and A1 skills
      let hasCompletedBasics = false;
      try {
        hasCompletedBasics = await this.checkUserSkillsCompleted();
        console.log(`[Immerse] buildPrompt: Skill check complete. Has completed basics: ${hasCompletedBasics}`);
      } catch (skillCheckError) {
        console.error('[Immerse] buildPrompt: Error checking skills, assuming not completed:', skillCheckError);
        hasCompletedBasics = false;
      }

      if (!hasCompletedBasics) {
        // Use basic Spanish prompt
        console.log('[Immerse] buildPrompt: User has NOT completed A0/A1 - attempting to use Basic Spanish prompt');

        let basicPrompt = null;
        try {
          basicPrompt = await this.loadBasicSpanishPrompt();
        } catch (loadError) {
          console.error('[Immerse] buildPrompt: Failed to load basic Spanish prompt file:', loadError);
        }

        if (basicPrompt) {
          console.log('[Immerse] buildPrompt: Successfully loaded Basic Spanish prompt, building final prompt...');
          let prompt = `${basicPrompt}\n\nTranslate the following numbered items to BASIC SPANISH. Output format: [number] translation (no bold, no markdown)\n\n`;
          elementsToTranslate.forEach(item => {
            prompt += `[${item.idx}] ${item.text}\n`;
          });
          console.log(`[Immerse] buildPrompt: Built Basic Spanish prompt (${prompt.length} chars)`);
          return prompt;
        } else {
          console.log('[Immerse] buildPrompt: Failed to load basic prompt, falling back to standard CEFR prompt');
        }
      } else {
        console.log('[Immerse] buildPrompt: User HAS completed A0/A1 - using standard CEFR prompt');
      }

      // Standard CEFR-based prompt
      console.log(`[Immerse] buildPrompt: Building standard CEFR ${userLevel} prompt`);
      let prompt = `Translate each numbered item to ${targetLang} at CEFR level ${userLevel}. Use vocabulary and grammar appropriate for ${userLevel} proficiency. Keep translations natural and at the right complexity for a ${userLevel} learner. Output format: [number] ${targetLang} text (no bold, no markdown)\n\n`;
      elementsToTranslate.forEach(item => {
        prompt += `[${item.idx}] ${item.text}\n`;
      });
      console.log(`[Immerse] buildPrompt: Built standard prompt (${prompt.length} chars)`);
      return prompt;
    } catch (error) {
      console.error('[Immerse] buildPrompt: Unexpected error, building fallback prompt:', error);
      // Fallback: return a simple standard prompt
      let fallbackPrompt = `Translate each numbered item to ${targetLang}. Output format: [number] translation\n\n`;
      elementsToTranslate.forEach(item => {
        fallbackPrompt += `[${item.idx}] ${item.text}\n`;
      });
      return fallbackPrompt;
    }
  }

  getLangCode(languageName) {
    const langMap = {
      'Spanish': 'es',
      'French': 'fr',
      'German': 'de',
      'Italian': 'it',
      'Portuguese': 'pt',
      'Dutch': 'nl',
      'Russian': 'ru',
      'Japanese': 'ja',
      'Korean': 'ko',
      'Chinese': 'zh',
      'Arabic': 'ar',
      'Hindi': 'hi',
      'Turkish': 'tr',
      'Polish': 'pl',
      'Swedish': 'sv',
      'Norwegian': 'no',
      'Danish': 'da',
      'Finnish': 'fi',
      'Greek': 'el',
      'Czech': 'cs',
      'Hungarian': 'hu',
      'Romanian': 'ro',
      'Thai': 'th',
      'Vietnamese': 'vi',
      'Indonesian': 'id',
      'Malay': 'ms',
      'Filipino': 'fil',
      'Hebrew': 'he',
      'Ukrainian': 'uk',
      'Bengali': 'bn',
      'Persian': 'fa',
      'Urdu': 'ur',
      'Swahili': 'sw'
    };
    return langMap[languageName] || 'es';
  }

  reset() {
    console.log('[Immerse] Reset called');
    
    // Abort current job
    if (this.abortController) {
      console.log('[Immerse] Aborting current job...');
      this.abortController.abort();
    }
    
    // Destroy session
    if (this.currentSession?.destroy) {
      console.log('[Immerse] Destroying session...');
      this.currentSession.destroy();
    }
    
    // Stop observer
    if (this.observer) {
      this.observer.disconnect();
      this.observer = null;
    }
    
    // Clear state
    this.elements = [];
    this.applied = new Set();
    this.translations = new Map();
    this.fullContext = '';
    this.jobStarted = false;
    this.jobComplete = false;
    this.liveMode = false;
    this.isApplying = false;
    this.isTranslatingNew = false;
    this.abortController = null;
    this.currentSession = null;
    this.seenTexts = new Set();
    
    console.log('[Immerse] State cleared, starting new job in 800ms');
    
    // Start pre-translation on new page after short delay
    setTimeout(() => this.runJob(), 800);
  }

  async checkAPI() {
    if (!window.LanguageModel) {
      return { available: false };
    }
    return { 
      available: true,
      liveMode: this.liveMode,
      jobStarted: this.jobStarted,
      jobComplete: this.jobComplete,
      isTranslatingNew: this.isTranslatingNew,
      applied: this.applied.size,
      cached: this.translations.size,
      total: this.elements.length
    };
  }

  collectContent(scanOnly = false) {
    if (this.elements.length > 0 && !scanOnly) return;
    
    let idx = this.elements.length;
    const newElements = [];
    
    // Skip these elements entirely
    const skipTags = new Set(['SCRIPT', 'STYLE', 'NOSCRIPT', 'IFRAME', 'SVG', 'PATH', 'CODE', 'PRE', 'TEXTAREA', 'INPUT', 'BUTTON', 'SELECT', 'OPTION']);
    const skipSelectors = 'nav, footer, aside, [role="navigation"], [role="banner"], .nav, .menu, .footer, .sidebar, .ad, .advertisement, .social, .share, .cookie, .popup, .modal, [aria-hidden="true"]';
    
    // Helper to check if element is visible
    const isVisible = (el) => {
      if (!el || !el.getBoundingClientRect) return false;
      const rect = el.getBoundingClientRect();
      if (rect.width === 0 || rect.height === 0) return false;
      const style = getComputedStyle(el);
      return style.display !== 'none' && style.visibility !== 'hidden' && style.opacity !== '0';
    };
    
    // Helper to add element
    const addElement = (node) => {
      if (!node || !node.textContent) return null;
      if (node.hasAttribute('data-immerse-id')) return null;
      if (node.closest(skipSelectors)) return null;
      if (node.closest('script, style, noscript, svg, code, pre')) return null;
      
      const text = node.textContent.trim();
      if (!text || text.length < 10 || text.length > 2000) return null;
      
      // Skip if looks like code/script
      if (text.includes('function ') || text.includes('const ') || text.includes('var ') ||
          text.includes('=>') || text.includes('});') || text.includes('.src =') ||
          text.includes('document.') || text.includes('window.')) {
        return null;
      }
      
      // Skip duplicates
      const textKey = text.substring(0, 60);
      if (this.seenTexts.has(textKey)) return null;
      this.seenTexts.add(textKey);
      
      // Skip if not visible
      if (!isVisible(node)) return null;
      
      const { text: extractedText, links, hasLinks } = this.extractContent(node);
      node.setAttribute('data-immerse-id', idx);
      
      const element = { 
        el: node, 
        text: extractedText, 
        links, 
        hasLinks, 
        tag: node.tagName,
        originalText: text,
        idx: idx
      };
      
      this.elements.push(element);
      newElements.push(element);
      idx++;
      return element;
    };
    
    // APPROACH: Walk all text nodes, find their meaningful parents
    const textNodes = [];
    const walker = document.createTreeWalker(
      document.body,
      NodeFilter.SHOW_TEXT,
      {
        acceptNode: (node) => {
          const text = node.textContent?.trim();
          if (!text || text.length < 10) return NodeFilter.FILTER_REJECT;
          
          const parent = node.parentElement;
          if (!parent) return NodeFilter.FILTER_REJECT;
          if (skipTags.has(parent.tagName)) return NodeFilter.FILTER_REJECT;
          
          // Check ancestors for script/style/etc
          if (parent.closest('script, style, noscript, svg, code, pre, textarea')) {
            return NodeFilter.FILTER_REJECT;
          }
          
          // Skip if looks like code
          if (text.includes('function ') || text.includes('const ') || text.includes('var ') || 
              text.includes('=>') || text.includes('});') || text.includes('img.src')) {
            return NodeFilter.FILTER_REJECT;
          }
          
          return NodeFilter.FILTER_ACCEPT;
        }
      }
    );
    
    let node;
    while ((node = walker.nextNode())) {
      textNodes.push(node);
    }
    
    console.log(`[Immerse] Found ${textNodes.length} text nodes`);
    
    // Group text nodes by their block-level parent
    const processed = new Set();
    const blockTags = new Set(['P', 'H1', 'H2', 'H3', 'H4', 'H5', 'H6', 'LI', 'TD', 'TH', 'BLOCKQUOTE', 'FIGCAPTION', 'CAPTION', 'ARTICLE', 'SECTION']);
    
    for (const textNode of textNodes) {
      let el = textNode.parentElement;
      
      // Walk up to find the best container (block-level or meaningful)
      let depth = 0;
      while (el && depth < 8) {
        // Found a good block-level element
        if (blockTags.has(el.tagName)) break;
        
        // Check for semantic classes that indicate content
        const className = el.className?.toString() || '';
        if (className.match(/headline|title|body|content|paragraph|summary|description|text/i)) break;
        
        // Don't go past article/main containers
        if (el.tagName === 'ARTICLE' || el.tagName === 'MAIN') break;
        
        if (el.parentElement && el.parentElement.tagName !== 'BODY') {
          el = el.parentElement;
        } else {
          break;
        }
        depth++;
      }
      
      if (!el || processed.has(el)) continue;
      
      // Skip if this element has too many block children (it's a container, not content)
      const blockChildren = el.querySelectorAll('p, h1, h2, h3, h4, h5, h6, li, article, section').length;
      if (blockChildren > 2) continue;
      
      processed.add(el);
      addElement(el);
    }
    
    // Full context from elements
    this.fullContext = this.elements.slice(0, 30).map(e => e.originalText || e.text).join('\n\n');
    if (this.fullContext.length > 8000) {
      this.fullContext = this.fullContext.substring(0, 8000);
    }
    
    if (!scanOnly) {
      console.log(`[Immerse] Collected ${this.elements.length} elements`);
      if (this.elements.length > 0) {
        console.log('[Immerse] Found elements:', this.elements.map(e => e.tag + ': ' + e.text));
      }
    }
    
    return newElements;
  }
  
  // Scan for new elements and translate them if in live mode
  async scanAndTranslateNew() {
    if (this.isTranslatingNew) return; // Already translating new elements
    
    const newElements = this.collectContent(true);
    
    if (newElements.length === 0) return;
    
    console.log(`[Immerse] Detected ${newElements.length} new elements, starting translation...`);
    
    if (!this.liveMode) return;
    
    this.isTranslatingNew = true;
    
    try {
      await this.translateElements(newElements);
    } finally {
      this.isTranslatingNew = false;
    }
  }
  
  // Translate a specific set of elements
  async translateElements(elementsToTranslate) {
    if (elementsToTranslate.length === 0) return;

    try {
      // Get user data from storage
      const userData = await chrome.storage.local.get('immerse_user');
      const user = userData.immerse_user;
      const targetLang = user?.target_language || 'Spanish';
      const targetLangCode = this.getLangCode(targetLang);
      const userLevel = user?.current_level || 'A0';

      const session = await LanguageModel.create({
        expectedInputLanguages: ['en'],
        expectedOutputLanguages: [targetLangCode]
      });

      // Sort: longer ones first, short ones (<20 chars) last
      const sortedElements = [...elementsToTranslate].sort((a, b) => {
        const aShort = a.text.length < 20;
        const bShort = b.text.length < 20;
        if (aShort && !bShort) return 1;
        if (!aShort && bShort) return -1;
        return 0;
      });

      // Build prompt based on user's skill completion
      const prompt = await this.buildPrompt(targetLang, targetLangCode, userLevel, sortedElements);
      
      console.log(`[Immerse] Translating ${elementsToTranslate.length} new elements...`);

      // Start timer for LLM latency
      const startTime = performance.now();
      const stream = await session.promptStreaming(prompt);
      let accumulated = '';
      let firstChunkReceived = false;

      for await (const chunk of stream) {
        // Measure time to first chunk
        if (!firstChunkReceived) {
          const latency = performance.now() - startTime;
          console.log(`[Immerse] ⚡ First chunk received in ${latency.toFixed(0)}ms`);
          firstChunkReceived = true;
        }

        const chunkText = typeof chunk === 'string' ? chunk : String(chunk);
        if (chunkText.length > accumulated.length) {
          accumulated = chunkText;
        } else {
          accumulated += chunkText;
        }

        // Parse for these specific elements
        this.parseNewElements(accumulated, elementsToTranslate);
        this.applyAllCached();
      }
      
      // Final parse
      this.parseNewElements(accumulated, elementsToTranslate);
      this.applyAllCached();
      
      if (session.destroy) session.destroy();
      
      console.log(`[Immerse] New elements translation complete`);
      
    } catch (err) {
      console.error('[Immerse] Error translating new elements:', err);
    }
  }
  
  // Parse translations for specific elements
  parseNewElements(text, elements) {
    for (const item of elements) {
      const idx = item.idx;
      if (this.translations.has(idx)) continue;
      
      const markerRegex = new RegExp(`\\*?\\*?\\[${idx}\\]\\s*`);
      const match = text.match(markerRegex);
      if (!match) continue;
      
      const start = match.index + match[0].length;
      
      // Find next marker (any number)
      const nextMatch = text.slice(start).match(/\*?\*?\[\d+\]/);
      
      let end = nextMatch ? start + nextMatch.index : text.length;
      
      let translation = text.substring(start, end).trim();
      
      // Clean up
      translation = translation.replace(/\*?\*?\[\d+\]\s*/g, '');
      translation = translation.replace(/\*\*/g, '');
      translation = translation.trim();
      
      if (translation.length > 3 && translation.length < item.text.length * 3) {
        this.translations.set(idx, translation);
        console.log(`[Immerse] Cached new [${idx}]: ${translation.substring(0, 40)}...`);
      }
    }
  }

  extractContent(el) {
    const links = [];
    let linkIndex = 0;
    const clone = el.cloneNode(true);
    
    clone.querySelectorAll('a').forEach(a => {
      links.push({ index: linkIndex, href: a.href });
      a.textContent = `[[L${linkIndex}]]` + a.textContent + `[[/L${linkIndex}]]`;
      linkIndex++;
    });
    
    return { text: clone.textContent.trim(), links, hasLinks: links.length > 0 };
  }

  reconstructHTML(translation, links) {
    if (links.length === 0) return translation;
    
    let html = translation;
    links.forEach(link => {
      const regex = new RegExp(`\\[\\[L${link.index}\\]\\](.*?)\\[\\[/L${link.index}\\]\\]`, 'g');
      html = html.replace(regex, `<a href="${link.href}">$1</a>`);
    });
    
    html = html.replace(/\[\[L\d+\]\]/g, '').replace(/\[\[\/L\d+\]\]/g, '');
    return html;
  }

  applyTranslation(idx) {
    if (this.applied.has(idx)) return;
    
    const translation = this.translations.get(idx);
    // Find element by idx property, not array position
    const item = this.elements.find(e => e.idx === idx);
    if (!item || !translation) return;
    
    // Check if element is still in DOM, if not try to re-find it
    let el = item.el;
    if (!el.isConnected) {
      console.log(`[Immerse] Element [${idx}] disconnected, re-finding...`);
      el = document.querySelector(`[data-immerse-id="${idx}"]`);
      if (el) {
        item.el = el;
        console.log(`[Immerse] Re-found element [${idx}]`);
      } else {
        console.log(`[Immerse] Could not re-find element [${idx}]`);
        return;
      }
    }
    
    this.isApplying = true;
    
    if (item.hasLinks) {
      el.innerHTML = this.reconstructHTML(translation, item.links);
    } else {
      el.textContent = translation;
    }
    
    // Visual feedback
    el.style.backgroundColor = 'rgba(76, 175, 80, 0.2)';
    setTimeout(() => { el.style.backgroundColor = ''; }, 400);
    
    this.isApplying = false;
    this.applied.add(idx);
    console.log(`[Immerse] Applied [${idx}]`);
  }
  
  // Re-apply all translations (for React sites that re-render)
  reapplyAll() {
    if (this.isApplying) return;
    this.isApplying = true;
    
    let reapplied = 0;
    this.translations.forEach((translation, idx) => {
      const item = this.elements.find(e => e.idx === idx);
      if (!item) return;
      
      // First try by data attribute
      let el = document.querySelector(`[data-immerse-id="${idx}"]`);
      
      // If not found or disconnected, try to find by original text content
      if (!el || !el.isConnected) {
        const searchText = item.originalStart;
        if (!searchText) return;
        
        try {
          const candidates = document.querySelectorAll(item.tag);
          for (const candidate of candidates) {
            const existingId = candidate.getAttribute('data-immerse-id');
            if (existingId && existingId !== String(idx)) continue;
            
            const candidateText = candidate.textContent.trim();
            if (candidateText.startsWith(searchText)) {
              el = candidate;
              el.setAttribute('data-immerse-id', idx);
              item.el = el;
              break;
            }
          }
        } catch (e) {}
      }
      
      if (el && el.isConnected) {
        const currentText = el.textContent.trim();
        const translationStart = translation.substring(0, 30);
        // Only apply if current text is NOT already translated
        if (!currentText.startsWith(translationStart)) {
          if (item.hasLinks) {
            el.innerHTML = this.reconstructHTML(translation, item.links);
          } else {
            el.textContent = translation;
          }
          reapplied++;
        }
      }
    });
    
    this.isApplying = false;
    
    if (reapplied > 0) {
      console.log(`[Immerse] Re-applied ${reapplied} translations`);
    }
  }

  async runJob() {
    if (this.jobStarted) return;
    this.jobStarted = true;
    
    // Create abort controller for this job
    this.abortController = new AbortController();
    const signal = this.abortController.signal;
    
    await new Promise(r => setTimeout(r, 500));
    
    // Check if aborted
    if (signal.aborted) {
      console.log('[Immerse] Job aborted before collect');
      return;
    }
    
    this.collectContent();
    
    console.log(`[Immerse] Collected ${this.elements.length} elements`);
    if (this.elements.length > 0) {
      console.log('[Immerse] First element:', this.elements[0].text.substring(0, 60));
    }
    
    // If no elements found, retry a few times (SPA content might still be loading)
    if (this.elements.length === 0) {
      for (let retry = 0; retry < 5; retry++) {
        console.log(`[Immerse] No elements found, retry ${retry + 1}/5...`);
        await new Promise(r => setTimeout(r, 500));
        
        if (signal.aborted) {
          console.log('[Immerse] Job aborted during retry');
          return;
        }
        
        this.elements = []; // Clear for re-collect
        this.collectContent();
        
        if (this.elements.length > 0) {
          console.log(`[Immerse] Found ${this.elements.length} elements on retry ${retry + 1}`);
          break;
        }
      }
    }
    
    if (this.elements.length === 0) {
      console.log('[Immerse] No elements found after retries, marking complete');
      this.jobComplete = true;
      return;
    }
    
    try {
      // Check if aborted
      if (signal.aborted) {
        console.log('[Immerse] Job aborted before session');
        return;
      }

      // Get user data from storage
      const userData = await chrome.storage.local.get('immerse_user');
      const user = userData.immerse_user;
      const targetLang = user?.target_language || 'Spanish';
      const targetLangCode = this.getLangCode(targetLang);
      const userLevel = user?.current_level || 'A0';

      console.log(`[Immerse] Creating session for ${targetLang} (${targetLangCode}), user level: ${userLevel}...`);
      const session = await LanguageModel.create({
        expectedInputLanguages: ['en'],
        expectedOutputLanguages: [targetLangCode]
      });
      this.currentSession = session;
      console.log('[Immerse] Session created');

      // Check if aborted
      if (signal.aborted) {
        console.log('[Immerse] Job aborted after session');
        if (session.destroy) session.destroy();
        return;
      }

      // Sort elements: longer ones first, short ones (<20 chars) last
      const sortedElements = [...this.elements].sort((a, b) => {
        const aShort = a.text.length < 20;
        const bShort = b.text.length < 20;
        if (aShort && !bShort) return 1;  // a goes last
        if (!aShort && bShort) return -1; // b goes last
        return 0; // preserve original order within groups
      });

      // Build prompt based on user's skill completion
      const prompt = await this.buildPrompt(targetLang, targetLangCode, userLevel, sortedElements);

      console.log(`[Immerse] Prompt: ${prompt.length} chars for ${this.elements.length} elements`);

      // Start timer for LLM latency
      const startTime = performance.now();
      const stream = await session.promptStreaming(prompt);
      console.log('[Immerse] Stream started, waiting for response...');
      let accumulated = '';
      let chunkCount = 0;
      let lastUpdate = Date.now();
      let firstChunkReceived = false;

      for await (const chunk of stream) {
        // Check if aborted during streaming
        if (signal.aborted) {
          console.log('[Immerse] Job aborted during stream');
          if (session.destroy) session.destroy();
          return;
        }

        chunkCount++;

        // Measure time to first chunk
        if (!firstChunkReceived) {
          const latency = performance.now() - startTime;
          console.log(`[Immerse] ⚡ First chunk received in ${latency.toFixed(0)}ms (${(latency / 1000).toFixed(2)}s)`);
          firstChunkReceived = true;
        }

        const chunkText = typeof chunk === 'string' ? chunk : String(chunk);
        
        // Detect if API returns incremental or full accumulated text
        if (chunkText.length > accumulated.length) {
          accumulated = chunkText;
        } else {
          accumulated += chunkText;
        }
        
        lastUpdate = Date.now();
        
        // Log more chunks to understand streaming behavior
        if (chunkCount <= 10 || chunkCount % 20 === 0) {
          console.log(`[Immerse] Chunk ${chunkCount}: +${chunkText.length} chars, total: ${accumulated.length}, cached: ${this.translations.size}`);
        }
        
        const beforeCache = this.translations.size;
        this.parseAndCache(accumulated);
        
        // Log when new translations are found
        if (this.translations.size > beforeCache) {
          console.log(`[Immerse] New translations found: ${this.translations.size} total`);
        }
        
        // Fallback apply in case parseAndCache didn't apply all
        if (this.liveMode) {
          this.applyAllCached();
        }
      }
      
      console.log(`[Immerse] Stream complete. Chunks: ${chunkCount}, Total: ${accumulated.length} chars`);
      console.log(`[Immerse] Translations cached: ${this.translations.size}/${this.elements.length}`);
      if (accumulated.length > 0) {
        console.log('[Immerse] Response preview:', accumulated.substring(0, 1000));
      } else {
        console.log('[Immerse] ERROR: Model returned empty response!');
      }
      
      // Final parse
      this.parseAndCache(accumulated);
      if (this.liveMode) {
        this.applyAllCached();
      }
      
      if (session.destroy) session.destroy();
      this.currentSession = null;
      
    } catch (err) {
      console.error('[Immerse] Error:', err);
      this.currentSession = null;
    }
    
    // Only set complete if not aborted (prevents race condition with reset)
    if (signal.aborted) {
      console.log('[Immerse] Job was aborted, not setting complete');
      return;
    }
    
    // Only mark complete if we actually translated something (or there was nothing to translate)
    if (this.translations.size > 0 || this.elements.length === 0) {
      this.jobComplete = true;
      console.log(`[Immerse] Job complete. Cached: ${this.translations.size}/${this.elements.length}`);
    } else {
      console.log(`[Immerse] Job finished but no translations cached. Elements: ${this.elements.length}. Allowing retry.`);
      // Reset so user can retry
      this.jobStarted = false;
      this.elements = [];
      this.applied = new Set();
      this.seenTexts = new Set();
    }
    
    if (this.liveMode) {
      this.applyAllCached();
    }
  }

  parseAndCache(text) {
    // Simple approach: find all [N] markers and extract text between them
    for (let i = 0; i < this.elements.length; i++) {
      const item = this.elements[i];
      const idx = item.idx;
      if (this.translations.has(idx)) continue;
      
      // Find this marker - just look for [N] anywhere
      const markerRegex = new RegExp(`\\[${idx}\\]`);
      const match = text.match(markerRegex);
      if (!match) continue;
      
      const markerEnd = match.index + match[0].length;
      
      // Find next marker (any number)
      const restOfText = text.slice(markerEnd);
      const nextMatch = restOfText.match(/\[\d+\]/);
      
      const isLast = i === this.elements.length - 1;
      
      // For non-last elements, need next marker to confirm this one is complete
      if (!isLast && !nextMatch) continue;
      
      let end = nextMatch ? markerEnd + nextMatch.index : text.length;
      let translation = text.substring(markerEnd, end).trim();
      
      // Clean up
      translation = translation.replace(/\*\*/g, ''); // Remove bold
      translation = translation.replace(/\[\d+\]/g, ''); // Remove stray markers
      translation = translation.trim();
      
      // Validate
      if (translation.length > 3 && translation.length < item.text.length * 3) {
        this.translations.set(idx, translation);
        console.log(`[Immerse] Cached [${idx}]: ${translation.substring(0, 50)}...`);
      }
    }
  }

  applyAllCached() {
    // Apply all available translations directly (no waiting for order)
    this.translations.forEach((_, idx) => {
      if (!this.applied.has(idx)) {
        this.applyTranslation(idx);
      }
    });
  }

  activate() {
    this.liveMode = true;
    this.applyAllCached();
    if (!this.jobStarted) {
      this.runJob();
    }
    
    // Watch for DOM changes - both for re-renders and new content
    if (!this.observer) {
      let debounceTimer = null;
      let scanDebounceTimer = null;
      
      this.observer = new MutationObserver((mutations) => {
        // Check if any new nodes were added that might need translation
        let hasNewNodes = false;
        for (const mutation of mutations) {
          if (mutation.addedNodes.length > 0) {
            for (const node of mutation.addedNodes) {
              if (node.nodeType === Node.ELEMENT_NODE) {
                // Check if this node or its children contain translatable elements
                const hasContent = node.matches?.('h1, h2, h3, p') || 
                                   node.querySelector?.('h1, h2, h3, p');
                if (hasContent) {
                  hasNewNodes = true;
                  break;
                }
              }
            }
          }
          if (hasNewNodes) break;
        }
        
        // Re-apply existing translations (for React re-renders)
        if (this.translations.size > 0 && !this.isApplying) {
          clearTimeout(debounceTimer);
          debounceTimer = setTimeout(() => this.reapplyAll(), 200);
        }
        
        // Scan for new elements if new content was added
        if (hasNewNodes && this.liveMode && !this.isApplying) {
          clearTimeout(scanDebounceTimer);
          scanDebounceTimer = setTimeout(() => this.scanAndTranslateNew(), 500);
        }
      });
      
      this.observer.observe(document.body, {
        childList: true,
        subtree: true
      });
      console.log('[Immerse] MutationObserver started (watching for new content)');
    }
  }
}

const translator = new PageTranslator();

// Detect SPA navigation (Reddit, etc.)
let lastUrl = location.href;
let urlCheckTimeout = null;

async function checkUrlChange() {
  if (location.href !== lastUrl) {
    console.log('[Immerse] URL changed:', lastUrl, '->', location.href);
    lastUrl = location.href;
    translator.reset();

    // Check if auto-translate is enabled for new URL
    const currentDomain = window.location.hostname;
    const data = await chrome.storage.local.get('auto_translate_domains');
    const autoTranslateDomains = data.auto_translate_domains || [];

    if (autoTranslateDomains.includes(currentDomain)) {
      console.log(`[Immerse] Auto-translate enabled for ${currentDomain} after navigation`);
      const userData = await chrome.storage.local.get('immerse_user');
      if (userData.immerse_user) {
        setTimeout(() => translator.activate(), 1000);
      }
    }
  }
}

// Watch for DOM mutations - throttled check
const urlObserver = new MutationObserver(() => {
  if (!urlCheckTimeout) {
    urlCheckTimeout = setTimeout(() => {
      checkUrlChange();
      urlCheckTimeout = null;
    }, 300);
  }
});
urlObserver.observe(document.body, { childList: true, subtree: true });

// Listen for popstate (back/forward)
window.addEventListener('popstate', checkUrlChange);

// Intercept pushState and replaceState
const originalPushState = history.pushState;
history.pushState = function(...args) {
  originalPushState.apply(this, args);
  setTimeout(checkUrlChange, 50);
};

const originalReplaceState = history.replaceState;
history.replaceState = function(...args) {
  originalReplaceState.apply(this, args);
  setTimeout(checkUrlChange, 50);
};

const init = async () => {
  // Check if auto-translate is enabled for this domain
  const currentDomain = window.location.hostname;
  const data = await chrome.storage.local.get('auto_translate_domains');
  const autoTranslateDomains = data.auto_translate_domains || [];

  if (autoTranslateDomains.includes(currentDomain)) {
    console.log(`[Immerse] Auto-translate enabled for ${currentDomain}, activating...`);
    // Wait for user data to be available
    const userData = await chrome.storage.local.get('immerse_user');
    if (userData.immerse_user) {
      setTimeout(() => translator.activate(), 500);
    } else {
      console.log('[Immerse] User not logged in, skipping auto-translate');
      setTimeout(() => translator.runJob(), 500);
    }
  } else {
    setTimeout(() => translator.runJob(), 500);
  }
};

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', init);
} else {
  init();
}

chrome.runtime.onMessage.addListener((msg, sender, resp) => {
  if (msg.type === 'CHECK_API_FROM_POPUP') {
    translator.checkAPI().then(resp);
    return true;
  }
  if (msg.type === 'START_TRANSLATION' || msg.type === 'START_TRANSLATION_FROM_POPUP') {
    translator.activate();
    resp({ success: true });
    return true;
  }
  if (msg.type === 'DOWNLOAD_MODEL') {
    LanguageModel.create().then(s => {
      if (s.destroy) s.destroy();
      resp({ success: true });
    }).catch(e => resp({ success: false, message: e.message }));
    return true;
  }
});
