const API_BASE = 'https://unlacerated-brett-unjudgeable.ngrok-free.dev';

// ====================
// AUTHENTICATION
// ====================
async function checkAuth() {
  const userData = await chrome.storage.local.get('immerse_user');
  return userData.immerse_user || null;
}

async function showView(viewName) {
  const loginView = document.getElementById('loginView');
  const dashboardView = document.getElementById('dashboardView');

  if (viewName === 'login') {
    loginView.classList.remove('hidden');
    dashboardView.classList.add('hidden');
  } else {
    loginView.classList.add('hidden');
    dashboardView.classList.remove('hidden');
  }
}

async function handleLogin(email, password) {
  const loginBtn = document.getElementById('loginBtn');
  const loginError = document.getElementById('loginError');

  loginBtn.disabled = true;
  loginBtn.textContent = 'Signing in...';
  loginError.classList.add('hidden');

  try {
    const response = await fetch(`${API_BASE}/api/login`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'ngrok-skip-browser-warning': 'true'
      },
      body: JSON.stringify({ email, password })
    });

    let data;
    try {
      data = await response.json();
    } catch (jsonError) {
      throw new Error('Could not connect to server');
    }

    if (!response.ok) {
      throw new Error(data.message || 'Login failed');
    }

    // Store user data
    await chrome.storage.local.set({ immerse_user: data.user });

    // Reload all tabs to activate extension
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      if (tab.url && !tab.url.startsWith('chrome://') && !tab.url.startsWith('chrome-extension://')) {
        chrome.tabs.reload(tab.id);
      }
    }

    // Show dashboard
    await initDashboard();

  } catch (error) {
    if (error.name === 'TypeError' && error.message.includes('fetch')) {
      loginError.textContent = 'Could not connect to server';
    } else {
      loginError.textContent = error.message;
    }
    loginError.classList.remove('hidden');
    loginBtn.disabled = false;
    loginBtn.textContent = 'Sign In';
  }
}

async function handleLogout() {
  await chrome.storage.local.remove('immerse_user');

  // Reload all tabs to deactivate extension
  const tabs = await chrome.tabs.query({});
  for (const tab of tabs) {
    if (tab.url && !tab.url.startsWith('chrome://') && !tab.url.startsWith('chrome-extension://')) {
      chrome.tabs.reload(tab.id);
    }
  }

  await showView('login');
}

async function initDashboard() {
  const user = await checkAuth();
  if (!user) {
    await showView('login');
    return;
  }

  await showView('dashboard');

  // Update user info display
  document.getElementById('userName').textContent = user.username;
  document.getElementById('userLevel').textContent = `${user.current_level} • ${user.target_language}`;

  // Update translate button text with user's target language
  const translateBtn = document.getElementById('translateBtn');
  translateBtn.textContent = `Translate to ${user.target_language}`;

  // Update auto-translate button
  await updateAutoTranslateButton();

  // Start translation UI updates
  await updateTranslationUI();
  setInterval(updateTranslationUI, 1000);
}

// ====================
// TRANSLATION UI (Original functionality preserved)
// ====================
let tab = null;
let activated = false;
let retryCount = 0;

async function getTab() {
  if (!tab) {
    const [t] = await chrome.tabs.query({ active: true, currentWindow: true });
    tab = t;
  }
  return tab;
}

async function sendMessage(message) {
  const t = await getTab();
  return chrome.tabs.sendMessage(t.id, message);
}

async function updateTranslationUI() {
  const translateBtn = document.getElementById('translateBtn');
  const statusText = document.getElementById('statusText');
  const statusIndicator = document.getElementById('statusIndicator');
  const errorDiv = document.getElementById('error');

  try {
    const t = await getTab();

    if (!t || t.url?.startsWith('chrome://') || t.url?.startsWith('chrome-extension://')) {
      setStatus('Cannot run here', 'error');
      translateBtn.disabled = true;
      return;
    }

    let state;
    try {
      state = await sendMessage({ type: 'CHECK_API_FROM_POPUP' });
    } catch (e) {
      // Content script not ready yet
      retryCount++;
      if (retryCount < 5) {
        setStatus('Loading...', 'working');
        translateBtn.disabled = true;
      } else {
        setStatus('Refresh page', 'error');
        translateBtn.disabled = true;
      }
      return;
    }

    retryCount = 0; // Reset on success

    if (!state || !state.available) {
      setStatus('AI not available', 'error');
      translateBtn.disabled = true;
      errorDiv.textContent = 'Chrome AI not found. Enable chrome://flags/#optimization-guide-on-device-model';
      errorDiv.classList.remove('hidden');
      return;
    }

    errorDiv.classList.add('hidden');

    // Sync activated state with content script
    if (!state.liveMode && activated) {
      // Content script was reset, reset popup too
      activated = false;
      translateBtn.style.background = '';
    }

    if (activated || state.liveMode) {
      activated = true;
      translateBtn.disabled = true;

      if (state.jobComplete && state.cached > 0 && !state.isTranslatingNew) {
        setStatus(`Translated ${state.applied}/${state.total}`, 'success');
        translateBtn.textContent = 'Translated ✓';
        translateBtn.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
      } else if (state.jobComplete && state.cached === 0 && !state.isTranslatingNew) {
        // Job finished but nothing translated - allow retry
        setStatus('No content found', 'error');
        translateBtn.textContent = 'Retry';
        translateBtn.disabled = false;
        activated = false;
      } else if (state.isTranslatingNew) {
        setStatus(`Translating new content...`, 'working');
        translateBtn.textContent = `${state.applied}/${state.total}`;
        translateBtn.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
      } else {
        setStatus(`Translating ${state.applied || 0}/${state.total || '?'}`, 'working');
        translateBtn.textContent = 'Translating...';
        translateBtn.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
      }
    } else {
      translateBtn.disabled = false;
      translateBtn.style.background = '';
      const user = await checkAuth();
      if (state.total > 0) {
        setStatus(`Ready (${state.total} items)`, 'ready');
      } else if (state.jobStarted) {
        setStatus('Scanning...', 'working');
      } else {
        setStatus('Ready', 'ready');
      }
      if (user) {
        translateBtn.textContent = `Translate to ${user.target_language}`;
      }
    }
  } catch (err) {
    console.error('Popup error:', err);
    setStatus('Error', 'error');
    translateBtn.disabled = true;
  }
}

function setStatus(text, state) {
  const statusText = document.getElementById('statusText');
  const statusIndicator = document.getElementById('statusIndicator');
  statusText.textContent = text;
  statusIndicator.className = 'status-indicator ' + state;
}

// ====================
// AUTO-TRANSLATE
// ====================
function getDomainFromUrl(url) {
  try {
    const urlObj = new URL(url);
    return urlObj.hostname;
  } catch (e) {
    return null;
  }
}

async function getAutoTranslateDomains() {
  const data = await chrome.storage.local.get('auto_translate_domains');
  return data.auto_translate_domains || [];
}

async function isAutoTranslateEnabled(domain) {
  const domains = await getAutoTranslateDomains();
  return domains.includes(domain);
}

async function toggleAutoTranslate(domain) {
  let domains = await getAutoTranslateDomains();

  if (domains.includes(domain)) {
    // Remove domain
    domains = domains.filter(d => d !== domain);
  } else {
    // Add domain
    domains.push(domain);
  }

  await chrome.storage.local.set({ auto_translate_domains: domains });
  return domains.includes(domain);
}

async function updateAutoTranslateButton() {
  const autoTranslateBtn = document.getElementById('autoTranslateBtn');
  const autoTranslateText = document.getElementById('autoTranslateText');
  const autoTranslateIcon = document.getElementById('autoTranslateIcon');

  if (!autoTranslateBtn) return;

  const t = await getTab();

  if (!t || !t.url || t.url.startsWith('chrome://') || t.url.startsWith('chrome-extension://')) {
    autoTranslateBtn.disabled = true;
    autoTranslateText.textContent = 'Not available here';
    return;
  }

  const domain = getDomainFromUrl(t.url);
  if (!domain) {
    autoTranslateBtn.disabled = true;
    return;
  }

  const isEnabled = await isAutoTranslateEnabled(domain);

  if (isEnabled) {
    autoTranslateBtn.classList.add('enabled');
    autoTranslateIcon.textContent = '✓';
    autoTranslateText.textContent = `Auto-translate ${domain}`;
  } else {
    autoTranslateBtn.classList.remove('enabled');
    autoTranslateIcon.textContent = '⚡';
    autoTranslateText.textContent = `Auto-translate ${domain}`;
  }

  autoTranslateBtn.disabled = false;
}

// ====================
// EVENT LISTENERS
// ====================
document.addEventListener('DOMContentLoaded', async () => {
  // Check authentication
  const user = await checkAuth();

  if (user) {
    await initDashboard();
  } else {
    await showView('login');
  }

  // Login form
  document.getElementById('loginForm').addEventListener('submit', async (e) => {
    e.preventDefault();
    const email = document.getElementById('email').value;
    const password = document.getElementById('password').value;
    await handleLogin(email, password);
  });

  // Logout button
  document.getElementById('logoutBtn').addEventListener('click', async () => {
    await handleLogout();
  });

  // Register link
  document.getElementById('registerLink').addEventListener('click', (e) => {
    e.preventDefault();
    chrome.tabs.create({ url: `${API_BASE}/register` });
  });

  // Translate button
  document.getElementById('translateBtn').addEventListener('click', async () => {
    if (activated) return;

    activated = true;
    const translateBtn = document.getElementById('translateBtn');
    translateBtn.disabled = true;
    translateBtn.textContent = 'Translating...';
    setStatus('Starting...', 'working');

    try {
      await sendMessage({ type: 'START_TRANSLATION_FROM_POPUP' });
    } catch (err) {
      console.error('Start error:', err);
    }

    updateTranslationUI();
  });

  // Auto-translate button
  document.getElementById('autoTranslateBtn')?.addEventListener('click', async () => {
    const t = await getTab();
    if (!t || !t.url) return;

    const domain = getDomainFromUrl(t.url);
    if (!domain) return;

    const isEnabled = await toggleAutoTranslate(domain);

    // Update button UI
    await updateAutoTranslateButton();

    // Show feedback
    const autoTranslateText = document.getElementById('autoTranslateText');
    if (isEnabled) {
      autoTranslateText.textContent = `✓ Enabled for ${domain}`;
      setTimeout(() => updateAutoTranslateButton(), 2000);
    } else {
      autoTranslateText.textContent = `Disabled for ${domain}`;
      setTimeout(() => updateAutoTranslateButton(), 2000);
    }
  });

  // Settings and about links
  document.getElementById('settingsLink')?.addEventListener('click', e => e.preventDefault());
  document.getElementById('aboutLink')?.addEventListener('click', e => e.preventDefault());
});
