// Content script - runs in the context of web pages
// This is a filler script that calls translate.js functionality

console.log('[Immerse] Content script loaded');

// Translation state
let translationState = {
  isEnabled: false,        // Whether translation is active on this page
  isAutoEnabled: false,    // Whether auto-translate is enabled for this domain
  hasUserRequested: false, // Whether user manually requested translation
  currentUser: null        // User data from popup
};

/*setTimeout(() => {
  var elems = scan();
  let combinedElems = [];
  let currentCombined = '';
  let currentIndex = 0;

  for (let i = 0; i < elems.length; i++) {
    let elemText = elems[i].text + ' ' || ''; // if text content for this element is falsy, we append '' to avoid error (idk when this would happen)

    if (currentCombined === '') {
      currentCombined = `[${currentIndex}]${elemText}`;
      currentIndex = i;
    } else {
      currentCombined += elemText;
    }

    // If combined text is >= 100 chars or this is the last element, save it
    if (currentCombined.length >= 100 || i === elems.length - 1) {
      combinedElems.push(currentCombined);
      currentCombined = '';
    }
  }

  // Concatenate all combined elements into one large string
  let result = combinedElems.join('');
  console.log(result);
}, 4500);
*/

/**
 * Check if translation should run based on current state
 * Returns true if either auto-translate is enabled OR user has requested translation
 */
function shouldTranslate() {
  return translationState.isEnabled &&
         (translationState.isAutoEnabled || translationState.hasUserRequested);
}

/**
 * Initialize translation state by checking auto-translate domains
 */
async function checkAutoTranslateStatus() {
  try {
    const currentDomain = window.location.hostname;

    // Use user-data.js helper to check auto-translate status
    translationState.isAutoEnabled = await isAutoTranslateEnabledForDomain(currentDomain);

    if (translationState.isAutoEnabled) {
      console.log('[Immerse] Auto-translate enabled for domain:', currentDomain);
    }

    return translationState.isAutoEnabled;
  } catch (error) {
    console.error('[Immerse] Error checking auto-translate status:', error);
    return false;
  }
}

/**
 * Start translation conditionally based on state
 */
async function conditionalTranslate(update=false) {
  if (!shouldTranslate()) {
    console.log('[Immerse] Translation not enabled - skipping');
    return;
  }

  console.log('[Immerse] Running conditional translation...');
  await (translate_again = safeTranslateLoaded()); // Safe wrapper that prevents infinite loops

  if (update && translate_again) {
    conditionalTranslate(update=true);
  }

  console.log('[Immerse] Starting mutation observer for dynamic content...');
  startObserver(); // Start watching for new content
}

// Listen for messages from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  console.log('[Immerse] Message received:', message.type);

  if (message.type === 'START_TRANSLATION_FROM_POPUP') {
    // User manually requested translation
    translationState.hasUserRequested = true;
    translationState.isEnabled = true;
    translationState.currentUser = message.user;

    console.log('[Immerse] Translation requested by user:', message.user?.username);

    // Start translation
    conditionalTranslate().then(() => {
      sendResponse({ status: 'success' });
    }).catch((error) => {
      console.error('[Immerse] Translation error:', error);
      sendResponse({ status: 'error', message: error.message });
    });

    return true; // Keep channel open for async response
  }

  if (message.type === 'CHECK_API_FROM_POPUP') {
    // Popup is checking translation state for UI updates
    const elems = typeof scan === 'function' ? scan() : [];
    sendResponse({
      liveMode: translationState.isEnabled,
      jobComplete: false,
      cached: 0,
      isTranslatingNew: false,
      applied: 0,
      total: elems.length,
      jobStarted: false
    });
    return true;
  }

  if (message.type === 'CHECK_TRANSLATION_STATUS') {
    // Popup is checking if translation is enabled
    sendResponse({
      isEnabled: translationState.isEnabled,
      isAutoEnabled: translationState.isAutoEnabled,
      hasUserRequested: translationState.hasUserRequested
    });
    return true;
  }

  if (message.type === 'LOGOUT') {
    // Reset translation state on logout
    translationState.isEnabled = false;
    translationState.hasUserRequested = false;
    translationState.currentUser = null;

    console.log('[Immerse] Translation state reset after logout');
    sendResponse({ status: 'success' });
    return true;
  }

  sendResponse({ status: 'received' });
  return true;
});

console.log('[Immerse] Content script initialization complete');

// Check auto-translate status on page load and start translation if enabled
setTimeout(async () => {
  const isAutoEnabled = await checkAutoTranslateStatus();

  if (isAutoEnabled) {
    // Auto-translate is enabled for this domain
    translationState.isEnabled = true;

    // Get user data from storage for auto-translate using user-data.js helper
    const user = await getUserData();
    if (user) {
      translationState.currentUser = user;

      console.log('[Immerse] Auto-translating with user:', user.username);
      await conditionalTranslate(true);
    } else {
      console.log('[Immerse] Auto-translate enabled but no user logged in');
    }
  } else {
    console.log('[Immerse] Auto-translate not enabled - waiting for user action');
  }
}, 1)



