// elements.js - DOM mutation detection and observation

console.log('[Immerse] elements.js loading...');

let mutationTimer = null;
let isTranslating = false;
let observer = null;

const DEBOUNCE_MS = 500; // Wait 500ms for mutations to settle
const MIN_TEXT_LENGTH = 100; // Minimum characters for significant content

/**
 * Helper function to extract all text content from a node and its children
 */

function scan(root = document.body) {
    const segments = [];
    const walker = document.createTreeWalker(root, NodeFilter.SHOW_TEXT);
    let node, id = 0;
  
    while ((node = walker.nextNode())) {
      const text = node.textContent?.trim();
      const parent = node.parentElement;
      if (!text || text.length < 10 || !parent || parent.closest(SKIP)) continue;
  
      parent.dataset.txId = String(id);
      segments.push({ id: id++, node, parent, text });
    }
    return segments;
  }

function getNodeTextContent(node) {
  if (node.nodeType === Node.TEXT_NODE) {
    return node.textContent?.trim() || '';
  }

  if (node.nodeType === Node.ELEMENT_NODE) {
    return node.textContent?.trim() || '';
  }

  return '';
}

/**
 * Check if mutations contain significant new text content
 * Returns true if at least one new text element with >= 100 characters is found
 */
function isSignificantChange(mutations) {
  for (const mutation of mutations) {
    if (mutation.type !== 'childList' || mutation.addedNodes.length === 0) {
      continue;
    }

    // Check each added node for significant text
    for (const node of mutation.addedNodes) {
      const textContent = getNodeTextContent(node);

      if (textContent.length >= MIN_TEXT_LENGTH) {
        console.log(`[Immerse] Found significant text node: ${textContent.length} chars`);
        return true;
      }
    }
  }

  return false;
}

/**
 * Start observing DOM mutations
 * Calls translate_loaded() when new content is detected
 */
function startObserver() {
  if (observer) {
    console.log('[Immerse] Observer already running');
    return;
  }

  console.log('[Immerse] Starting MutationObserver...');

  observer = new MutationObserver((mutations) => {
    // Ignore mutations that happen during translation
    if (isTranslating) {
      console.log('[Immerse] Ignoring mutations during translation');
      return;
    }

    // Check if mutations contain significant new text content
    if (!isSignificantChange(mutations)) {
      return;
    }

    console.log('[Immerse] Significant content detected, debouncing translation...');

    // Debounce: wait for mutations to settle before translating
    clearTimeout(mutationTimer);
    mutationTimer = setTimeout(() => {
      console.log('[Immerse] Mutations settled, triggering translation');
      expected_tags = window.translate_loaded();
      console.log('expected tags: ' + expected_tags);
    }, DEBOUNCE_MS);
  });

  // Start observing the document body for changes
  observer.observe(document.body, {
    childList: true,   // Watch for added/removed child nodes
    subtree: true,     // Watch the entire subtree
    characterData: false // Don't watch text changes (would see our own updates)
  });

  console.log('[Immerse] MutationObserver started');
}

/**
 * Stop observing DOM mutations
 */
function stopObserver() {
  if (observer) {
    observer.disconnect();
    observer = null;
    console.log('[Immerse] MutationObserver stopped');
  }
}

/**
 * Temporarily pause observation during translation to prevent infinite loops
 */
function pauseObserver() {
  if (observer) {
    observer.disconnect();
  }
}

/**
 * Resume observation after translation completes
 */
function resumeObserver() {
  if (observer) {
    observer.observe(document.body, {
      childList: true,
      subtree: true,
      characterData: false
    });
  }
}

/**
 * Wrapped translate_loaded that prevents infinite loops
 * This wraps the original translate_loaded function from translate.js
 * Note: translate_loaded is defined globally in translate.js (loaded before this file)
 */
async function safeTranslateLoaded() {
  if (isTranslating) {
    console.log('[Immerse] Translation already in progress, skipping');
    return;
  }

  isTranslating = true;
  pauseObserver(); // Stop watching during translation

  try {
    console.log('[Immerse] Starting safe translation...');
    // translate_loaded is defined in translate.js which loads before elements.js
    await window.translate_loaded();
    console.log('[Immerse] Safe translation completed');
  } catch (error) {
    console.error('[Immerse] Safe translation failed:', error);
  } finally {
    isTranslating = false;
    resumeObserver(); // Resume watching after translation
  }
}

function analyze_pageside_changes(expected_after, actual_after) { // using element additions as proxy for significant page changes. this misses changes to pre-existing elements, but is a good proxy for now and is easily modifiable because we have direct access to element data pre and post.
  if (actual_after.length - expected_after.length >= 2) {
    return true;
  }
  else return false;
}

// Expose functions globally
if (typeof window !== 'undefined') {
  window.startObserver = startObserver;
  window.stopObserver = stopObserver;
  window.safeTranslateLoaded = safeTranslateLoaded;
  window.analyze_pageside_changes = analyze_pageside_changes;
}

console.log('[Immerse] elements.js loaded');
