/**
 * USAGE EXAMPLES for GeminiTranslator
 *
 * This file shows how to use the gemini.js module to get streamed output from Google Gemini
 */

// ============================================================================
// EXAMPLE 1: Basic streaming with inline prompt
// ============================================================================
async function example1_BasicStreaming() {
  const gemini = new GeminiTranslator();

  // Check if Gemini is available
  const available = await gemini.isAvailable();
  if (!available) {
    console.error('Gemini is not available in this browser');
    return;
  }

  // Create a session
  await gemini.createSession({
    systemPrompt: 'You are a helpful translation assistant.'
  });

  // Stream a response with inline prompt
  const prompt = 'Translate "Hello, how are you?" to Spanish';

  console.log('Streaming response:');
  for await (const chunk of gemini.streamPrompt(prompt)) {
    console.log('Chunk:', chunk);
    // In a real app, you might update the UI here
  }

  // Clean up
  gemini.destroy();
}

// ============================================================================
// EXAMPLE 2: Streaming with prompt from a text file
// ============================================================================
async function example2_PromptFromFile() {
  const gemini = new GeminiTranslator();

  // Check availability
  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  // Create session
  await gemini.createSession();

  // Read prompt from a text file and stream the response
  // The file should be accessible via fetch (e.g., in the extension directory)
  const promptFilePath = 'prompts/translation-prompt.txt';

  try {
    for await (const chunk of gemini.streamPrompt(promptFilePath, { fromFile: true })) {
      console.log('Received:', chunk);
    }
  } catch (err) {
    console.error('Error streaming from file:', err);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 3: Streaming with callback for each chunk
// ============================================================================
async function example3_StreamingWithCallback() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  const prompt = 'Write a short poem about coding';

  // Use a callback to process each chunk
  const chunks = [];
  for await (const chunk of gemini.streamPrompt(prompt, {
    onChunk: (text) => {
      console.log('New chunk received, length:', text.length);
      // Update UI in real-time here
      updateUIWithText(text);
    }
  })) {
    chunks.push(chunk);
  }

  console.log('Final response:', chunks[chunks.length - 1]);
  gemini.destroy();
}

// Mock UI update function
function updateUIWithText(text) {
  // In a real Chrome extension, this might update a DOM element
  // For example: document.getElementById('output').textContent = text;
  console.log('UI updated with:', text.substring(0, 50) + '...');
}

// ============================================================================
// EXAMPLE 4: Non-streaming (get complete response)
// ============================================================================
async function example4_CompleteResponse() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  const prompt = 'Translate "Good morning" to Spanish';

  // Get the complete response (waits for all chunks)
  const response = await gemini.getResponse(prompt);
  console.log('Complete response:', response);

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 5: Translation use case (like in vALPHA)
// ============================================================================
async function example5_TranslationUseCase() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  // Create session for translation
  await gemini.createSession({
    systemPrompt: 'You are a translation assistant. Follow the instructions precisely and output only the requested format.'
  });

  // Build a translation prompt (similar to vALPHA style)
  const textsToTranslate = [
    { id: 1, text: 'Hello' },
    { id: 2, text: 'Goodbye' },
    { id: 3, text: 'Thank you' }
  ];

  const prompt = `Translate the following English texts to Spanish.

Input:
${textsToTranslate.map(item => `${item.id}. ${item.text}`).join('\n')}

Output format:
Return ONLY valid JSON in the following format:
[
  {"id": 1, "translation": "translated text"},
  {"id": 2, "translation": "translated text"}
]`;

  console.log('Translating...');
  let accumulated = '';

  for await (const chunk of gemini.streamPrompt(prompt)) {
    accumulated = chunk;
    console.log('Progress:', chunk.length, 'chars');
  }

  // Parse the final result
  try {
    const translations = JSON.parse(accumulated);
    console.log('Translations:', translations);
  } catch (err) {
    console.error('Failed to parse JSON:', err);
    console.log('Raw response:', accumulated);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 6: Using in a Chrome Extension Content Script
// ============================================================================
async function example6_ChromeExtensionUsage() {
  // In your content script (content.js), you can use it like this:

  const gemini = new GeminiTranslator();

  // Check if available
  const available = await gemini.isAvailable();
  if (!available) {
    console.log('Please enable Chrome AI in chrome://flags/#optimization-guide-on-device-model');
    return;
  }

  // Create session
  await gemini.createSession({
    systemPrompt: 'You are a helpful assistant for webpage translation.'
  });

  // Get selected text from the page
  const selectedText = window.getSelection().toString();

  if (selectedText) {
    const prompt = `Translate this to Spanish: "${selectedText}"`;

    // Stream the translation and show it in real-time
    let translation = '';
    for await (const chunk of gemini.streamPrompt(prompt)) {
      translation = chunk;
      // You could display this in a tooltip or overlay
      showTranslationTooltip(translation);
    }
  }

  // Don't forget to clean up when done
  // gemini.destroy();
}

function showTranslationTooltip(text) {
  // Mock function - implement your UI here
  console.log('Showing tooltip:', text);
}

// ============================================================================
// EXAMPLE 7: Combine predefined prompt with adaptive prompt
// ============================================================================
async function example7_CombinePrompts() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  // Predefined prompt (inline string)
  const predefinedPrompt = `You are a translation assistant.
Your task is to translate text to Spanish while maintaining the original meaning and tone.`;

  // Adaptive prompt with dynamic data
  const userText = 'Hello, how are you?';
  const targetLanguage = 'Spanish';
  const adaptivePrompt = `Translate the following to ${targetLanguage}: "${userText}"`;

  // Combine and stream
  for await (const chunk of gemini.streamCombinedPrompt(predefinedPrompt, adaptivePrompt)) {
    console.log('Translation chunk:', chunk);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 8: Combine prompt from file with adaptive prompt
// ============================================================================
async function example8_CombineFilePromptWithDynamic() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  // Predefined prompt from file
  const predefinedPrompt = 'prompts/translation-template.txt';

  // Dynamic content from user
  const textsToTranslate = [
    'Good morning',
    'Thank you very much',
    'See you later'
  ];

  const adaptivePrompt = `
Input texts:
${textsToTranslate.map((text, i) => `${i + 1}. ${text}`).join('\n')}

Please translate these to Spanish.`;

  // Combine and stream
  for await (const chunk of gemini.streamCombinedPrompt(predefinedPrompt, adaptivePrompt)) {
    console.log(chunk);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 9: Using template variables in combined prompts
// ============================================================================
async function example9_TemplateVariables() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  // Predefined prompt with placeholders
  const predefinedPrompt = `You are a {role} assistant.
Your expertise is in {expertise}.
Current user level: {userLevel}`;

  // Adaptive prompt with placeholders
  const adaptivePrompt = `User {userName} asks: {userQuestion}`;

  // Variables to replace
  const variables = {
    role: 'translation',
    expertise: 'Spanish language education',
    userLevel: 'A2',
    userName: 'Maria',
    userQuestion: 'How do I say "I love learning Spanish" in Spanish?'
  };

  // Combine with variable replacement
  for await (const chunk of gemini.streamCombinedPrompt(
    predefinedPrompt,
    adaptivePrompt,
    { variables }
  )) {
    console.log(chunk);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 10: Custom separator for combined prompts
// ============================================================================
async function example10_CustomSeparator() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  const predefinedPrompt = 'System: You are a helpful translator.';
  const adaptivePrompt = 'User: Translate "goodbye" to Spanish.';

  // Use custom separator
  for await (const chunk of gemini.streamCombinedPrompt(
    predefinedPrompt,
    adaptivePrompt,
    { separator: '\n---\n' }
  )) {
    console.log(chunk);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 11: Real-world use case - Translation with user context
// ============================================================================
async function example11_RealWorldTranslation() {
  const gemini = new GeminiTranslator();

  if (!(await gemini.isAvailable())) {
    console.error('Gemini not available');
    return;
  }

  await gemini.createSession();

  // Predefined prompt (could be from a file)
  const predefinedPrompt = `You are a Spanish translation assistant for language learners.
Translate to Spanish appropriate for level {level}.
Format: Return ONLY valid JSON array with translations.`;

  // User data (dynamic)
  const userData = {
    level: 'A1',
    completedSkills: ['greetings', 'numbers', 'colors']
  };

  // Content to translate (dynamic)
  const pageContent = [
    { id: 1, text: 'Welcome to our website' },
    { id: 2, text: 'Click here to continue' },
    { id: 3, text: 'Learn more about us' }
  ];

  const adaptivePrompt = `
User completed skills: {completedSkills}

Translate these texts:
${pageContent.map(item => `${item.id}. ${item.text}`).join('\n')}

Return format:
[{"id": 1, "translation": "..."}, ...]`;

  // Variables to replace
  const variables = {
    level: userData.level,
    completedSkills: userData.completedSkills.join(', ')
  };

  console.log('Starting translation with user context...');
  let accumulated = '';

  for await (const chunk of gemini.streamCombinedPrompt(
    predefinedPrompt,
    adaptivePrompt,
    {
      variables,
      onChunk: (text) => {
        console.log('Progress:', text.length, 'characters');
      }
    }
  )) {
    accumulated = chunk;
  }

  // Parse result
  try {
    const translations = JSON.parse(accumulated);
    console.log('Translations:', translations);

    // Apply to page content
    translations.forEach(t => {
      const item = pageContent.find(p => p.id === t.id);
      if (item) {
        console.log(`${item.text} → ${t.translation}`);
      }
    });
  } catch (err) {
    console.error('Failed to parse:', err);
  }

  gemini.destroy();
}

// ============================================================================
// EXAMPLE 12: Just combine prompts without streaming (get final prompt)
// ============================================================================
async function example12_JustCombinePrompts() {
  const gemini = new GeminiTranslator();

  const predefinedPrompt = 'You are a translator.';
  const adaptivePrompt = 'Translate: {text}';

  // Just get the combined prompt without streaming
  const combinedPrompt = await gemini.combinePrompts(
    predefinedPrompt,
    adaptivePrompt,
    {
      variables: { text: 'Hello world' },
      separator: '\n\n---\n\n'
    }
  );

  console.log('Combined prompt:');
  console.log(combinedPrompt);

  // Now you can use it however you want
  // For example, send it to streamPrompt later
  if (await gemini.isAvailable()) {
    await gemini.createSession();
    for await (const chunk of gemini.streamPrompt(combinedPrompt)) {
      console.log(chunk);
    }
    gemini.destroy();
  }
}

// ============================================================================
// To run these examples, uncomment the one you want to test:
// ============================================================================

// example1_BasicStreaming();
// example2_PromptFromFile();
// example3_StreamingWithCallback();
// example4_CompleteResponse();
// example5_TranslationUseCase();
// example6_ChromeExtensionUsage();
// example7_CombinePrompts();
// example8_CombineFilePromptWithDynamic();
// example9_TemplateVariables();
// example10_CustomSeparator();
// example11_RealWorldTranslation();
// example12_JustCombinePrompts();
