/**
 * Google Gemini (Chrome Built-in AI) Module
 *
 * Provides streaming translation functionality using Chrome's built-in LanguageModel API.
 * This is the correct way to call Gemini as used in extension vALPHA.
 */

class GeminiTranslator {
  constructor() {
    this.currentSession = null;
  }

  /**
   * Check if Gemini is available in this browser
   * @returns {Promise<boolean>}
   */
  async isAvailable() {
    try {
      // Check if the LanguageModel API is available
      // Use globalThis to work in both window (web pages) and self (service workers)
      console.log('[Gemini] Checking availability...');
      console.log('[Gemini] globalThis.ai exists:', !!globalThis.ai);
      console.log('[Gemini] self exists:', typeof self !== 'undefined');
      console.log('[Gemini] self.ai exists:', typeof self !== 'undefined' && !!self.ai);

      const ai = globalThis.ai || (typeof self !== 'undefined' ? self.ai : null);

      if (!ai) {
        console.log('[Gemini] ai object is null or undefined');
        return false;
      }

      console.log('[Gemini] ai object exists:', !!ai);
      console.log('[Gemini] ai.languageModel exists:', !!ai.languageModel);

      if (!ai.languageModel) {
        console.log('[Gemini] LanguageModel API not available');
        return false;
      }

      // Try to create a test session
      const capabilities = await ai.languageModel.capabilities();
      console.log('[Gemini] Capabilities:', capabilities);

      return capabilities.available === 'readily' || capabilities.available === 'after-download';
    } catch (err) {
      console.error('[Gemini] Availability check failed:', err);
      return false;
    }
  }

  /**
   * Create a Gemini session
   * Matches the interface from extension vALPHA
   *
   * @param {Object} options - Configuration options
   * @param {Array<string>} options.expectedInputLanguages - Input languages (e.g., ['en'])
   * @param {Array<string>} options.expectedOutputLanguages - Output languages (e.g., ['es'])
   * @returns {Promise<Object>} Session object with promptStreaming method
   */
  async createSession(options = {}) {
    try {
      // Use globalThis to work in both window (web pages) and self (service workers)
      const ai = globalThis.ai || (typeof self !== 'undefined' ? self.ai : null);

      if (!ai || !ai.languageModel) {
        throw new Error('LanguageModel API not available. Make sure you have Chrome AI enabled.');
      }

      // Create session using Chrome's built-in AI
      const session = await ai.languageModel.create({
        systemPrompt: options.systemPrompt || 'You are a helpful assistant.',
        ...options
      });

      this.currentSession = session;
      console.log('[Gemini] Session created successfully');

      return session;
    } catch (err) {
      console.error('[Gemini] Session creation failed:', err);
      throw err;
    }
  }

  /**
   * Stream a response from Gemini
   * This function allows you to call Gemini with a prompt (optionally from a text file)
   *
   * @param {string} prompt - The prompt text or file path
   * @param {Object} options - Configuration options
   * @param {boolean} options.fromFile - If true, treat prompt as a file path
   * @param {Function} options.onChunk - Callback for each chunk received
   * @returns {AsyncGenerator<string>} Stream of response chunks
   */
  async *streamPrompt(prompt, options = {}) {
    try {
      let promptText = prompt;

      // If fromFile is true, read the prompt from a file
      if (options.fromFile) {
        promptText = await this.readPromptFromFile(prompt);
      }

      // Create session if not already created
      if (!this.currentSession) {
        await this.createSession(options);
      }

      console.log('[Gemini] Starting stream with prompt:', promptText.substring(0, 100) + '...');

      // Use promptStreaming method - this is how it was done in vALPHA
      const stream = this.currentSession.promptStreaming(promptText);
      let accumulated = '';

      // Iterate through the stream
      for await (const chunk of stream) {
        accumulated = chunk; // Gemini returns accumulated text

        // Call onChunk callback if provided
        if (options.onChunk) {
          options.onChunk(chunk);
        }

        yield chunk;
      }

      console.log('[Gemini] Stream completed, total length:', accumulated.length);
    } catch (err) {
      console.error('[Gemini] Streaming error:', err);
      throw err;
    }
  }

  /**
   * Read prompt from a text file
   * @param {string} filePath - Path to the text file
   * @returns {Promise<string>} File contents
   */
  async readPromptFromFile(filePath) {
    try {
      const response = await fetch(filePath);
      if (!response.ok) {
        throw new Error(`Failed to read file: ${response.statusText}`);
      }
      const text = await response.text();
      console.log('[Gemini] Loaded prompt from file:', filePath);
      return text;
    } catch (err) {
      console.error('[Gemini] Failed to read prompt file:', err);
      throw err;
    }
  }

  /**
   * Combine predefined prompt with adaptive/dynamic prompt
   * Useful for having a template prompt and filling in dynamic data
   *
   * @param {string|Object} predefinedPrompt - Predefined prompt (string or object with filePath)
   * @param {string|Object} adaptivePrompt - Dynamic prompt to append (string or object with variables)
   * @param {Object} options - Configuration options
   * @param {string} options.separator - Separator between prompts (default: '\n\n')
   * @param {Object} options.variables - Variables to replace in template (e.g., {userName: 'John'})
   * @returns {Promise<string>} Combined prompt
   */
  async combinePrompts(predefinedPrompt, adaptivePrompt, options = {}) {
    const separator = options.separator || '\n\n';
    let predefinedText = '';
    let adaptiveText = '';

    // Handle predefined prompt
    if (typeof predefinedPrompt === 'string') {
      // Check if it's a file path
      if (predefinedPrompt.includes('.txt') || predefinedPrompt.includes('/')) {
        predefinedText = await this.readPromptFromFile(predefinedPrompt);
      } else {
        predefinedText = predefinedPrompt;
      }
    } else if (predefinedPrompt && predefinedPrompt.filePath) {
      predefinedText = await this.readPromptFromFile(predefinedPrompt.filePath);
    }

    // Handle adaptive prompt
    if (typeof adaptivePrompt === 'string') {
      adaptiveText = adaptivePrompt;
    } else if (adaptivePrompt && adaptivePrompt.template) {
      // Template with variables
      adaptiveText = adaptivePrompt.template;
    }

    // Replace variables if provided
    if (options.variables) {
      const replaceVariables = (text, vars) => {
        let result = text;
        for (const [key, value] of Object.entries(vars)) {
          // Support both {{variable}} and {variable} syntax
          result = result.replace(new RegExp(`{{${key}}}`, 'g'), value);
          result = result.replace(new RegExp(`{${key}}`, 'g'), value);
        }
        return result;
      };

      predefinedText = replaceVariables(predefinedText, options.variables);
      adaptiveText = replaceVariables(adaptiveText, options.variables);
    }

    // Combine the prompts
    const combined = predefinedText + separator + adaptiveText;
    console.log('[Gemini] Combined prompt length:', combined.length);

    return combined;
  }

  /**
   * Stream a response with combined prompts
   * Convenience method that combines prompts and streams the result
   *
   * @param {string|Object} predefinedPrompt - Predefined prompt
   * @param {string|Object} adaptivePrompt - Dynamic prompt
   * @param {Object} options - Configuration options
   * @returns {AsyncGenerator<string>} Stream of response chunks
   */
  async *streamCombinedPrompt(predefinedPrompt, adaptivePrompt, options = {}) {
    const combinedPrompt = await this.combinePrompts(predefinedPrompt, adaptivePrompt, options);

    // Create session if not already created
    if (!this.currentSession) {
      await this.createSession(options);
    }

    // Stream the combined prompt
    for await (const chunk of this.streamPrompt(combinedPrompt, options)) {
      yield chunk;
    }
  }

  /**
   * Simple helper: Get a complete response (non-streaming)
   * @param {string} prompt - The prompt text or file path
   * @param {Object} options - Configuration options
   * @returns {Promise<string>} Complete response
   */
  async getResponse(prompt, options = {}) {
    let fullResponse = '';

    for await (const chunk of this.streamPrompt(prompt, options)) {
      fullResponse = chunk; // Gemini returns accumulated text
    }

    return fullResponse;
  }

  /**
   * Destroy the current session
   * Important for cleanup
   */
  destroy() {
    if (this.currentSession && this.currentSession.destroy) {
      this.currentSession.destroy();
      console.log('[Gemini] Session destroyed');
    }
    this.currentSession = null;
  }
}

// Export for use in content scripts and other modules
if (typeof window !== 'undefined') {
  window.GeminiTranslator = GeminiTranslator;
}

// Also support module exports for modern JavaScript
if (typeof module !== 'undefined' && module.exports) {
  module.exports = GeminiTranslator;
}

// TRANSLATION EXECUTION:

export async function TranslatePage() {
    const gemini = new GeminiTranslator();

    if (!(await gemini.isAvailable())) {
      console.error('[Gemini] Gemini not available');
      return;
    }

    console.log('[Gemini] Creating session...');
    await gemini.createSession();

    // Predefined prompt from file - use chrome.runtime.getURL for service worker context
    const predefinedPrompt = chrome.runtime.getURL('prompts/translation-template.txt');

    // Dynamic content from user
    const textsToTranslate = [
      'Good morning',
      'Thank you very much',
      'See you later'
    ];

    const adaptivePrompt = `
  Input texts:
  ${textsToTranslate.map((text, i) => `${i + 1}. ${text}`).join('\n')}

  Please translate these to Spanish.`;

    // Combine and stream
    for await (const chunk of gemini.streamCombinedPrompt(predefinedPrompt, adaptivePrompt)) {
      console.log('BELOW IS CHUNK OF TRANSLATED TEXT:')
      console.log(chunk);
    }

    gemini.destroy();
  }

// Also export the GeminiTranslator class for ES6 modules
export default GeminiTranslator;