/**
 * OpenAI API Helper
 * Simple GPT function that takes text and a prompt file
 * Routes through our secure backend proxy to keep API key safe
 */

const OPENAI_API_URL = 'https://immersebackend-production.up.railway.app/api/openai/chat/completions';

/**
 * Main GPT function - streams response with chunk callback
 * @param {string} text - The text to process
 * @param {string} promptFile - Path to prompt file (relative to extension root)
 * @param {Function} onChunk - Callback function called with each chunk of text
 * @returns {Promise<string>} The complete response text
 */
async function gpt(text, promptFile, additionalSystemPrompt, onChunk) {
  const promptResponse = await fetch(chrome.runtime.getURL(promptFile));
  const systemPrompt = await promptResponse.text() + additionalSystemPrompt;

  const messages = [
    { role: 'system', content: systemPrompt},
    { role: 'user', content: text }
  ];

  try {
    const response = await fetch(OPENAI_API_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
        // No Authorization header needed - backend handles the API key
      },
      body: JSON.stringify({
        model: 'gpt-4.1',
        messages: messages,
        max_tokens: 9999,
        temperature: 0.7,
        stream: true
      })
    });

    if (!response.ok) {
      const errorData = await response.json();
      throw new Error(`OpenAI API error: ${errorData.error?.message || response.statusText}`);
    }

    const reader = response.body.getReader();
    const decoder = new TextDecoder();
    let fullResponse = '';

    while (true) {
      const { done, value } = await reader.read();
      if (done) break;

      const chunk = decoder.decode(value);
      const lines = chunk.split('\n').filter(line => line.trim() !== '');

      for (const line of lines) {
        if (line.startsWith('data: ')) {
          const data = line.slice(6);
          if (data === '[DONE]') continue;

          try {
            const parsed = JSON.parse(data);
            const content = parsed.choices[0]?.delta?.content;
            if (content) {
              fullResponse += content;
              if (onChunk) onChunk(content);
            }
          } catch (e) { // parsing error handling 4 incomplete chunks
          }
        }
      }
    }

    return fullResponse;
  } catch (error) {
    console.error('[OpenAI] Error:', error);
    throw error;
  }
}
