// API Base URL - Update this when deploying or changing ngrok tunnel
const API_BASE = 'https://immersebackend-production.up.railway.app';

// State
let currentUser = null;
let tab = null;
let activated = false;
let retryCount = 0;

document.addEventListener('DOMContentLoaded', async () => {
  // Initialize
  await loadUser();
  setupEventListeners();

  if (currentUser) {
    showTranslationView();
    await updateUI();
    setInterval(updateUI, 1000);
  } else {
    showLoginView();
  }
});

// ============================================================================
// Authentication Functions
// ============================================================================

async function loadUser() {
  const stored = await chrome.storage.local.get(['user']);
  if (stored.user) {
    currentUser = stored.user;
  }
}

async function saveUser(user) {
  currentUser = user;
  await chrome.storage.local.set({ user });
}

async function clearUser() {
  currentUser = null;
  await chrome.storage.local.remove(['user']);
}

async function login(email, password) {
  const response = await fetch(`${API_BASE}/api/login`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email, password })
  });

  const data = await response.json();

  if (!data.success) {
    throw new Error(data.message || 'Login failed');
  }

  return data.user;
}

// ============================================================================
// View Functions
// ============================================================================

function showLoginView() {
  document.getElementById('loginView').classList.remove('hidden');
  document.getElementById('translationView').classList.add('hidden');
}

function showTranslationView() {
  document.getElementById('loginView').classList.add('hidden');
  document.getElementById('translationView').classList.remove('hidden');

  if (currentUser) {
    document.getElementById('userName').textContent = currentUser.username;
    document.getElementById('userLevel').textContent = currentUser.current_level;
  }
}

// ============================================================================
// Event Listeners
// ============================================================================

function setupEventListeners() {
  // Login form
  document.getElementById('loginForm')?.addEventListener('submit', async (e) => {
    e.preventDefault();

    const email = document.getElementById('email').value;
    const password = document.getElementById('password').value;
    const loginBtn = document.getElementById('loginBtn');
    const loginError = document.getElementById('loginError');

    loginBtn.disabled = true;
    loginBtn.textContent = 'Signing in...';
    loginError.classList.add('hidden');

    try {
      const user = await login(email, password);
      await saveUser(user);

      showTranslationView();
      await updateUI();
      setInterval(updateUI, 1000);
    } catch (err) {
      loginError.textContent = err.message;
      loginError.classList.remove('hidden');
    } finally {
      loginBtn.disabled = false;
      loginBtn.textContent = 'Sign In';
    }
  });

  // Logout button
  document.getElementById('logoutBtn')?.addEventListener('click', async () => {
    await clearUser();
    activated = false;
    showLoginView();

    // Reset translation state in content script
    try {
      await sendMessage({ type: 'LOGOUT' });
    } catch (e) {
      // Content script might not be loaded
    }
  });

  // Translate button
  document.getElementById('translateBtn')?.addEventListener('click', async () => {
    if (activated) return;

    activated = true;
    const translateBtn = document.getElementById('translateBtn');
    translateBtn.disabled = true;
    translateBtn.textContent = 'Translating...';
    setStatus('Starting...', 'working');

    try {
      // Send user data to content script
      await sendMessage({
        type: 'START_TRANSLATION_FROM_POPUP',
        user: currentUser
      });
    } catch (err) {
      console.error('Start error:', err);
    }

    updateUI();
  });

  // Auto-translate button
  document.getElementById('autoTranslateBtn')?.addEventListener('click', async () => {
    const t = await getTab();
    if (!t || !t.url) return;

    const domain = getDomainFromUrl(t.url);
    if (!domain) return;

    const stored = await chrome.storage.local.get(['autoTranslateDomains']);
    const domains = stored.autoTranslateDomains || [];

    const index = domains.indexOf(domain);
    if (index > -1) {
      // Remove domain
      domains.splice(index, 1);
    } else {
      // Add domain
      domains.push(domain);
    }

    await chrome.storage.local.set({ autoTranslateDomains: domains });
    updateAutoTranslateUI(domain, index === -1);
  });

  // Settings and About (placeholder)
  document.getElementById('settingsLink')?.addEventListener('click', e => {
    e.preventDefault();
    chrome.tabs.create({ url: `${API_BASE}/settings` });
  });

  document.getElementById('aboutLink')?.addEventListener('click', e => {
    e.preventDefault();
    chrome.tabs.create({ url: `${API_BASE}/` });
  });
}

// ============================================================================
// Translation UI Functions
// ============================================================================

async function getTab() {
  if (!tab) {
    const [t] = await chrome.tabs.query({ active: true, currentWindow: true });
    tab = t;
  }
  return tab;
}

async function sendMessage(message) {
  const t = await getTab();
  return chrome.tabs.sendMessage(t.id, message);
}

/**
 * Check translation status from content script
 * Returns object with isEnabled, isAutoEnabled, and hasUserRequested
 */
async function checkTranslationStatus() {
  try {
    const status = await sendMessage({ type: 'CHECK_TRANSLATION_STATUS' });
    console.log('[Popup] Translation status:', status);
    return status;
  } catch (error) {
    console.error('[Popup] Error checking translation status:', error);
    return {
      isEnabled: false,
      isAutoEnabled: false,
      hasUserRequested: false
    };
  }
}

async function updateUI() {
  if (!currentUser) return;

  try {
    const t = await getTab();
    const translateBtn = document.getElementById('translateBtn');
    const errorDiv = document.getElementById('error');

    if (!t || t.url?.startsWith('chrome://') || t.url?.startsWith('chrome-extension://')) {
      setStatus('Cannot run here', 'error');
      translateBtn.disabled = true;
      return;
    }

    // Check and update auto-translate status
    await checkAutoTranslateStatus();

    // Check current translation status from content script
    const translationStatus = await checkTranslationStatus();

    let state;
    try {
      state = await sendMessage({ type: 'CHECK_API_FROM_POPUP' });
    } catch (e) {
      // Content script not ready yet
      retryCount++;
      if (retryCount < 5) {
        setStatus('Loading...', 'working');
        translateBtn.disabled = true;
      } else {
        setStatus('Refresh page', 'error');
        translateBtn.disabled = true;
      }
      return;
    }

    retryCount = 0; // Reset on success

    // Hide any previous errors
    errorDiv.classList.add('hidden');

    // Sync activated state with content script
    if (!state.liveMode && activated) {
      // Content script was reset, reset popup too
      activated = false;
      translateBtn.style.background = '';
    }

    if (activated || state.liveMode) {
      // Translation is active
      activated = true;
      translateBtn.disabled = true;
      translateBtn.textContent = 'Translating...';
      translateBtn.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
      setStatus('Translating...', 'working');
    } else {
      // Translation not active - show translate button
      translateBtn.disabled = false;
      translateBtn.textContent = 'Translate to Spanish';
      translateBtn.style.background = '';
      setStatus('Ready', 'ready');
    }
  } catch (err) {
    console.error('Popup error:', err);
    setStatus('Error', 'error');
    document.getElementById('translateBtn').disabled = true;
  }
}

function setStatus(text, state) {
  const statusText = document.getElementById('statusText');
  const statusIndicator = document.getElementById('statusIndicator');

  if (statusText) statusText.textContent = text;
  if (statusIndicator) statusIndicator.className = 'status-indicator ' + state;
}

// ============================================================================
// Auto-translate Functions
// ============================================================================

function getDomainFromUrl(url) {
  try {
    const urlObj = new URL(url);
    return urlObj.hostname;
  } catch (e) {
    return null;
  }
}

async function updateAutoTranslateUI(domain, isEnabled) {
  const btn = document.getElementById('autoTranslateBtn');
  const text = document.getElementById('autoTranslateText');
  const info = document.getElementById('autoTranslateInfo');

  if (isEnabled) {
    btn.classList.add('active');
    text.textContent = 'Auto-translate Enabled';
    info.textContent = `Pages on ${domain} will translate automatically`;
    info.classList.add('active');
  } else {
    btn.classList.remove('active');
    text.textContent = 'Enable Auto-translate';
    info.textContent = `Click to auto-translate all pages on ${domain}`;
    info.classList.remove('active');
  }
}

async function checkAutoTranslateStatus() {
  const t = await getTab();
  if (!t || !t.url) return;

  const domain = getDomainFromUrl(t.url);
  if (!domain) return;

  const stored = await chrome.storage.local.get(['autoTranslateDomains']);
  const domains = stored.autoTranslateDomains || [];
  const isEnabled = domains.includes(domain);

  updateAutoTranslateUI(domain, isEnabled);

  // Auto-trigger translation if domain is in the list
  if (isEnabled && currentUser && !activated) {
    console.log('[Popup] Auto-translating domain:', domain);
    setTimeout(async () => {
      try {
        await sendMessage({
          type: 'START_TRANSLATION_FROM_POPUP',
          user: currentUser
        });
      } catch (err) {
        console.error('[Popup] Auto-translate error:', err);
      }
    }, 500);
  }
}
