/**
 * User Data Helper
 * Handles fetching and managing user account data from Chrome storage
 */

/**
 * Get the current user's account data from Chrome storage
 * @returns {Promise<Object|null>} User object with account info, or null if not logged in
 *
 * User object structure:
 * {
 *   id: number,
 *   username: string,
 *   email: string,
 *   current_level: string (A0, A1, A2, B1, B2, C1, C2),
 *   target_language: string,
 *   vocabulary_count: number,
 *   completed_skills: array of skill IDs
 * }
 */
async function getUserData() {
  try {
    const result = await chrome.storage.local.get(['user']);

    if (result.user) {
      console.log('[UserData] User found:', result.user.username);
      return result.user;
    } else {
      console.log('[UserData] No user logged in');
      return null;
    }
  } catch (error) {
    console.error('[UserData] Error fetching user data:', error);
    return null;
  }
}

/**
 * Get specific user properties
 * @returns {Promise<Object>} Object with individual user properties
 */
async function getUserProperties() {
  const user = await getUserData();

  if (!user) {
    return {
      id: null,
      username: null,
      email: null,
      current_level: 'A0',
      target_language: 'Spanish',
      vocabulary_count: 0,
      completed_skills: []
    };
  }

  return {
    id: user.id,
    username: user.username,
    email: user.email,
    current_level: user.current_level || 'A0',
    target_language: user.target_language || 'Spanish',
    vocabulary_count: user.vocabulary_count || 0,
    completed_skills: user.completed_skills || []
  };
}

/**
 * Check if user is logged in
 * @returns {Promise<boolean>} True if user is logged in
 */
async function isUserLoggedIn() {
  const user = await getUserData();
  return user !== null;
}


/**
 * Get the list of domains that have auto-translate enabled
 * @returns {Promise<string[]>} Array of domain names
 */
async function getAutoTranslateDomains() {
  try {
    const result = await chrome.storage.local.get(['autoTranslateDomains']);
    return result.autoTranslateDomains || [];
  } catch (error) {
    console.error('[UserData] Error fetching auto-translate domains:', error);
    return [];
  }
}

/**
 * Check if a specific domain has auto-translate enabled
 * @param {string} domain - The domain to check
 * @returns {Promise<boolean>} True if auto-translate is enabled for this domain
 */
async function isAutoTranslateEnabledForDomain(domain) {
  const domains = await getAutoTranslateDomains();
  return domains.includes(domain);
}

/**
 * Listen for user data updates from popup
 * @param {Function} callback - Called when user data is updated with new user object
 */
function onUserDataUpdate(callback) {
  chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
    if (message.type === 'START_TRANSLATION_FROM_POPUP' && message.user) {
      console.log('[UserData] User data received from popup:', message.user.username);
      callback(message.user);
    }
  });
}

// Export for use in other files
if (typeof module !== 'undefined' && module.exports) {
  module.exports = {
    getUserData,
    getUserProperties,
    isUserLoggedIn,
    getAutoTranslateDomains,
    isAutoTranslateEnabledForDomain,
    onUserDataUpdate
  };
}
